
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_scale.c */

#include "fs_itype.h"
#include "fs_fontscal.h"    /* fs_GlyphInfoType et al */
#include "fs_scratch.h"     /* FsScratchSpace object  */

FS_LONG check_scale(_DS_ TTF_HEAD *head, FS_FIXED s00, FS_FIXED s01, FS_FIXED s10, FS_FIXED s11);

/*lint -e578  Warning 578: Declaration of symbol '' hides symbol */

#if ( defined(FS_EDGE_RENDER) || defined(FS_GRAYMAPS) ) && (defined(FS_BITMAPS))
static FS_VOID get_gasp_behavior(SFNT *sfnt, FS_USHORT *gbehavior)
{
    TTF  *ttf;
    FS_ULONG xppm;
    LFNT *lfnt;
    SENV *senv;
    FS_USHORT numRanges;
    FS_USHORT rangeMaxPPEM;
    int i;
    FS_BYTE    *pGasp;
    lfnt = (LFNT *)(sfnt->lfnt);

    *gbehavior = 0xFFFF;
    if (lfnt->fnt_type == PFR_TYPE)
        return;

    ttf = (FS_VOID *)(lfnt->fnt);

    if (!ttf->gasp)    /* gasp not found */
        return;
    senv = sfnt->senv;

    xppm = senv->xppm;
    pGasp = (FS_BYTE *)(ttf->gasp);

    pGasp += 2; /* skip version */
    numRanges = GET_xWORD(pGasp);
    pGasp += 2;
    for ( i = 0; i < numRanges; i++ )
    {
        rangeMaxPPEM = GET_xWORD(pGasp);
        if (xppm > rangeMaxPPEM)
        {
            pGasp += 4;
            continue;
        }

        pGasp += 2;
        *gbehavior = GET_xWORD(pGasp);
        return;
    }
    return;
}
#endif
/****************************************************************/
/* set up <key> as needed for current TTF and SCALE */
/* nx = s00*x + s01*y; and ny = s10*x + s11*y */
/* should probably re-write these to exclude the ttin/ttout layer */
FS_LONG scale_font_ttf(_DS_ SFNT *sfnt, FS_FIXED s00, FS_FIXED s01, FS_FIXED s10, FS_FIXED s11)
{
#ifdef FS_RENDER
    fs_GlyphInputType *ttin;
    fs_GlyphInfoType  *ttout;
    fsg_SplineKey *key;
    FS_LONG error;
    FS_FIXED xppm, yppm, tan_s;
#endif
    LFNT *lfnt;
    TTF *ttf;

    lfnt = (LFNT *)(sfnt->lfnt);

    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return STATE.error;
    }
    ttf = (FS_VOID *)(lfnt->fnt);

    STATE.error = check_scale(_PS_ (TTF_HEAD *)(ttf->head), s00, s01, s10, s11);
    if (STATE.error)
    {
        return STATE.error;
    }

#ifdef FS_RENDER
    ttin = (fs_GlyphInputType *)FsScratchSpace_reserve(&STATE.server->scratch, FS_state_ptr,
            sizeof(fs_GlyphInputType));
    if (ttin == 0 || STATE.error) return STATE.error;
    SYS_MEMSET(ttin, 0, sizeof(fs_GlyphInputType));

    ttin->sfnt = sfnt;

    ttout = (fs_GlyphInfoType *)FsScratchSpace_reserve(&STATE.server->scratch, FS_state_ptr,
            sizeof(fs_GlyphInfoType));
    if (ttout == 0 || STATE.error)
    {
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
        return STATE.error;
    }

    ttout->memorySizes[KEY_PTR_BASE] = sizeof(fsg_SplineKey);

    /* allocate the SFNT ttkey, freed in delete_key() */
#ifdef FS_MEM_DBG
    STATE.memdbgid = "fsg_SplineKey";
#endif
    ttin->memoryBases[KEY_PTR_BASE] = FSS_calloc(_PS_ sizeof(fsg_SplineKey));
    if (0 == ttin->memoryBases[KEY_PTR_BASE])
    {
        return STATE.error;
    }

    key = (fsg_SplineKey *)(FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]);
    key->read_programs = 1;    /* they are as yet unread */
    key->sfnt = sfnt;
    key->maxp = ttf->maxp;

#ifdef FS_CFFR
    key->max_contours = ttf->maxp->maxContours;
    key->max_points = ttf->maxp->maxPoints;
#endif

    key->state = FS_INITIALIZED;
    ttin->param.newsfnt.platformID = 0xffff;

    STATE.error = fs_NewSfnt(ttin, ttout);
    if (STATE.error)
    {
        error = STATE.error;
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]));
        return STATE.error = error;
    }
    if (STATE.server->workspaceSize < ttout->memorySizes[WORK_SPACE_BASE])
    {
        /* resize the workspace */
        if (STATE.server->workspaceSize > 0)
        {
            FSS_free(_PS_ (FS_BYTE *)(STATE.server->workspace));
        }
        STATE.server->workspaceSize = ttout->memorySizes[WORK_SPACE_BASE];

#ifdef FS_MEM_DBG
        STATE.memdbgid = "workspace";
#endif
        STATE.server->workspace = FSS_calloc(_PS_ ttout->memorySizes[WORK_SPACE_BASE]);
        if (0 == STATE.server->workspace)
        {
            STATE.server->workspaceSize = 0;

            error = STATE.error;
            FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]));
            FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
            FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttout);
            return STATE.error = error;
        }
    }

    /* allocate the ttkey's private font space, freed in delete_key() */
#ifdef FS_MEM_DBG
    STATE.memdbgid = "PRIVATE_FONT_SPACE_BASE";
#endif
    ttin->memoryBases[PRIVATE_FONT_SPACE_BASE] = FSS_malloc(_PS_ ttout->memorySizes[PRIVATE_FONT_SPACE_BASE]);
    if (0 == ttin->memoryBases[PRIVATE_FONT_SPACE_BASE])
    {
        error = STATE.error;
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]));
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttout);
        return STATE.error = error;
    }

    /* process the transformation ... note:
     * new_x = m[0]*x + m[3]*y + m[6] and new_y = m[1]*x + m[4]*y + m[7]
     */
    {
        FS_FIXED *m = (FS_FIXED *)(&ttin->param.newtrans.matrix);
        m[0] = s00;
        m[3] = s01;
        m[1] = s10;
        m[4] = s11;
        m[2] = m[5] = m[6] = m[7] = 0;
        m[8] = FRACT_ONE;    /* why this is a FRACT_ONE and not FIXED_ONE is a mystery */
    }
    ttin->param.newtrans.pointSize = FIXED_ONE;
    ttin->param.newtrans.xResolution = 72;
    ttin->param.newtrans.yResolution = 72;
    ttin->param.newtrans.pixelDiameter = FIXEDSQRT2;

    /* new transformation (also runs FontProgram) */
    STATE.error = fs_NewTransformation(_PS_ ttin, ttout);
    if (STATE.error)
    {
        error = STATE.error;
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]));
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[PRIVATE_FONT_SPACE_BASE]));
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttout);
        return STATE.error = error;
    }
#ifdef FS_HINTS
    /* run the PreProgram -- if we are hinting */
    STATE.error = fsg_RunPreProgram(_PS_ key);
    if (STATE.error)
    {
        error = STATE.error;
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]));
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[PRIVATE_FONT_SPACE_BASE]));
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttout);
        return STATE.error = error;
    }
#endif

    STATE.server->workspace_sfnt = sfnt;  /* indicate sfnt using the workspace */

    /* set the <senv> */
#ifdef FS_MEM_DBG
    STATE.memdbgid = "SENV";
#endif
    sfnt->senv = (SENV *)FSS_calloc(_PS_ sizeof(SENV));
    if (sfnt->senv == 0)
    {
        error = STATE.error;
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]));
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[PRIVATE_FONT_SPACE_BASE]));
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttout);
        return STATE.error = error;
    }
    sfnt->senv->ttkey = (FS_BYTE *)key;

    /* vanilla transform -- no rotation,  oblique, rotation, mirroring */
    sfnt->senv->vanilla = (s00 > 0 && s01 == 0 && s10 == 0 && s11 > 0);

#ifdef FS_HINTS
    if (!sfnt->senv->vanilla)
        sfnt->rtgah_suitable = RTGAH_NOPE;
#endif

    /* decode the scale to get sizes and skew used to make scale */
    error = get_scale_inputs(sfnt->user_scale, &xppm, &yppm, &tan_s);
    if (error != SUCCESS)
    {
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[KEY_PTR_BASE]));
        FSS_free(_PS_ (FS_BYTE *)(ttin->memoryBases[PRIVATE_FONT_SPACE_BASE]));
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
        FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttout);
        return STATE.error = error;
    }

    sfnt->senv->xppm = FS_ROUND(xppm);
    sfnt->senv->yppm = FS_ROUND(yppm);

    {
        FS_BYTE render_bv;

        render_bv = 0;
        render_bv |= DOGRAY; /* DOGRAY is default on unless gasp table changes it **/
        render_bv |= GRIDFIT;    /* GRIDFIT is default on unless gasp table changes it */
#if ( defined(FS_EDGE_RENDER) || defined(FS_GRAYMAPS) ) && (defined(FS_BITMAPS))
        {
            FS_USHORT gasp_bv = 0;
            get_gasp_behavior(sfnt, &gasp_bv);
            if (gasp_bv != 0xFFFF) /* has gasp table */
            {
                gasp_bv &= 0x03;    /* turn off other bits in gasp */
                render_bv &= gasp_bv; /* turn off DOGRAY & GRIDFIT if they are off in gasp */
            }
        }
#endif
        render_bv |= DOEDGE; /* DOEDGE is default on */

#ifdef FS_EDGE_RENDER
#if (defined(FS_GRAYMAPS))
        if (!ttf->adfh) /* adfh table does not exist */
            render_bv &= ~DOEDGE;  /* turn off DOEDGE */
#endif
#endif

        sfnt->senv->render_behavior = render_bv;
    }
#ifdef FS_PSEUDO_BOLD
    {
        FS_LONG size;
        size = sfnt->bold_pct * STATE.lpm;
        sfnt->senv->bold_width = FS_ROUND(size);
    }
#endif

#ifdef FS_STIK
    /* calculate the stroke width in pixels */
    {
        FS_SHORT temp;
        FS_LONG size = sfnt->stroke_pct * STATE.lpm;
        temp = FS_ROUND(size);
        sfnt->senv->stroke_width = MAX(1, temp);
    }
#endif /* FS_STIK */


    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttin);
    FsScratchSpace_release(&STATE.server->scratch, FS_state_ptr, ttout);
#endif /* FS_RENDER */

    return STATE.error = SUCCESS;
}

/*******************************************************************/
/* this is for all the calls which require a valid SFNT, LFNT, TTF */
/* This function requires a current lfnt and sfnt and should       */
/* follow a call to map_char which sets these in the STATE         */
FS_LONG check_sfnt(_DS0_)
{
    SFNT *sfnt = STATE.cur_sfnt;
    LFNT *lfnt;
    SENV *senv;

    /* check the SFNT */
    if (sfnt == 0)
        return STATE.error = ERR_NO_CURRENT_SFNT;

    lfnt = (LFNT *)(sfnt->lfnt);

    /* check the LFNT */
    if (lfnt == 0)
        return STATE.error = ERR_NO_CURRENT_LFNT;

    /* reload the TTF ? */
    if (!lfnt->fnt && (load_fnt(_PS_ lfnt) != SUCCESS))
    {
        return STATE.error = ERR_BAD_LFNT;
    }

#ifdef FS_DEBUG
    /* make sure cur_lfnt is still loaded */
    for (lfnt = (LFNT *)(STATE.server->loaded_fonts); lfnt; lfnt = (LFNT *)(lfnt->next))
    {
        if (lfnt == STATE.cur_lfnt)
            break;
    }

    if (lfnt == 0)
    {
        FS_PRINTF(("*** STATE.cur_lfnt no longer in loaded_fonts list\n"));
        return STATE.error = ERR_BAD_LFNT;
    }

    /* make sure cur_sfnt is still loaded */
    for (sfnt = (SFNT *)(STATE.server->scaled_fonts); sfnt; sfnt = (SFNT *)(sfnt->next))
    {
        if (sfnt == STATE.cur_sfnt)
            break;
    }

    if (sfnt == 0)
    {
        FS_PRINTF(("*** STATE.cur_sfnt no longer in loaded_fonts list\n"));
        return STATE.error = ERR_BAD_SFNT;
    }
#endif

    /* check whether we need to reload the guts of the sfnt */
    /* another client could have removed it                 */
    senv = (SENV *)(sfnt->senv);
    if (senv == 0)
    {
        FS_FIXED *s = sfnt->user_scale;

        /* restore the guts of the sfnt */
        scale_font(_PS_ sfnt, s[0], s[1], s[2], s[3]);
        if (STATE.error)
        {
            FSS_free(_PS_ senv);
            return STATE.error;
        }

#if defined(FS_STIK) && defined(FS_RENDER)
        if ( ((lfnt->fontflags & FONTFLAG_STIK) == FONTFLAG_STIK) &&
             !(lfnt->fontflags & FONTFLAG_NEW_AA_ON))
        {
#ifdef FS_EDGE_HINTS
            TTF *ttf = (FS_VOID *)(lfnt->fnt);
            if (!ttf)
                return STATE.error = ERR_BAD_LFNT;
            if (ttf->adfh)
                return STATE.error = SUCCESS; /* ref lines not used with Edge fonts */
#endif
            /*** until stik fonts are properly hinted ***/
            /*** for STIK font autohinting, we need to ***/
            /*** get reflines from the characters "HOxo" ***/
            if ( sfnt->senv->vanilla &&
                 sfnt->senv->cap_round == 0 )
            {
                SENV *senv = sfnt->senv;
                FS_ULONG flags = STATE.flags;
                FS_USHORT platform = STATE.platform;
                FS_USHORT encoding = STATE.encoding;
                FS_USHORT font = STATE.cur_font;

                if (STATE.flags & FLAGS_CMAP_OFF)
                {
                    FS_set_flags(_PS_ FLAGS_CMAP_ON );
                }
                FSS_set_cmap(_PS_ 3, 1);
                senv->language = OTHER;
                FS_set_flags(_PS_ FLAGS_VERTICAL_OFF);

                if (STATE.cur_sfnt && (STATE.error == SUCCESS))
                {
                    FS_OUTLINE *outl;

                    outl = internal_get_outline(_PS_ (FS_USHORT)'H');
                    if (outl == 0 || outl->num == 0 || (outl->type[0] & OUTLINE_CHAR))
                    {
                        FSS_free_char(_PS_ outl);
                        outl = internal_get_outline(_PS_ 0xFF28);
                    }
                    if (outl)
                    {
                        senv->base_square = outl->lo_y;
                        senv->cap_square = outl->hi_y;
                        FSS_free_char(_PS_ outl);
                    }

                    outl = internal_get_outline(_PS_ (FS_USHORT)'O');
                    if (outl == 0 || outl->num == 0 || (outl->type[0] & OUTLINE_CHAR))
                    {
                        FSS_free_char(_PS_ outl);
                        outl = internal_get_outline(_PS_ 0xFF2F);
                    }
                    if (outl)
                    {
                        senv->base_round = outl->lo_y;
                        senv->cap_round = outl->hi_y;
                        FSS_free_char(_PS_ outl);
                    }

                    outl = internal_get_outline(_PS_ (FS_USHORT)'x');
                    if (outl == 0 || outl->num == 0 || (outl->type[0] & OUTLINE_CHAR))
                    {
                        FSS_free_char(_PS_ outl);
                        outl = internal_get_outline(_PS_ 0xFF58);
                    }
                    if (outl)
                    {
                        senv->x_square = outl->hi_y;
                        FSS_free_char(_PS_ outl);
                    }

                    outl = internal_get_outline(_PS_ (FS_USHORT)'o');
                    if (outl == 0 || outl->num == 0 || (outl->type[0] & OUTLINE_CHAR))
                    {
                        FSS_free_char(_PS_ outl);
                        outl = internal_get_outline(_PS_ 0xFF4F);
                    }
                    if (outl)
                    {
                        senv->x_round = outl->hi_y;
                        senv->lsb = outl->lo_x;
                        senv->rsb = outl->dx - outl->hi_x;
                        FSS_free_char(_PS_ outl);
                    }
                }
                STATE.flags = flags;
                FSS_set_cmap(_PS_ platform, encoding);
                STATE.cur_font = font;
                STATE.cur_lfnt = lfnt;
                STATE.cur_sfnt = sfnt;
            }
        }
#endif /* FS_STIK && FS_RENDER */
    }

    return STATE.error = SUCCESS;
}

/*lint +e578  Warning 578: Declaration of symbol '' hides symbol */

/*****************************************************************
 * get_scale_inputs() returns xppm, yppm, and tangent skew angle
 * used to produce a given set of scale transformation values.
 * Always returns positive xppm, yppm regardless of rotation or
 * mirror used. tan_s may be positive or negative.
 * rotation invariant values for xppm and yppm
 * x" = x'*cos(phi) - y'*sin(phi)
 * y" = x'*sin(phi) + y'*cos(phi)
 *
 * x" = (x*s00 + y*s01)*cos(phi) - (x*s10 + y*s11)*sin(phi)
 * y" = (x*s00 + y*s01)*sin(phi) + (x*s10 + y*s11)*cos(phi)
 *
 * x" = x*(s00*cos(phi)-s10*sin(phi)) + y*(s01*cos(phi)-s11*sin(phi))
 * y" = x*(s00*sin(phi)+s10*cos(phi)) + y*(s01*sin(phi)+s11*cos(phi))
 *
 * to find xppm, need to zero the coeff of y in x"
 * (s01*cos(phi) - s11*sin(phi)) = 0
 * (s01*cos(phi) = s11*sin(phi))
 * s01/s11 = sin(phi)/cos(phi)
 * v[0] = s01, v[1] = s11, normalize(v) == sin(phi)/cos(phi)
 * xppm = s00*v[1] - s10*v[0]
 *
 * to find yppm, we need to zero the coeff of x in y"
 * (s00*sin(phi)+s10*cos(phi)) = 0
 * s00*sin(phi) = -s10*cos(phi)
 * sin(phi)/cos(phi) = -s10/s00
 * v[0] = -s10, v[1] = s00; normalize(v)
 * yppm = (s01*sin(phi)+s11*cos(phi)) = s01*v[0]+s11*v[1]
 */
FS_LONG get_scale_inputs(FS_FIXED *scale, FS_FIXED *xppm, FS_FIXED *yppm, FS_FIXED *tan_s)
{
    FS_FIXED s00, s01, s10, s11;
    FIXED_VECTOR v;
    FS_FIXED lim;

    /* check for vanilla transform first */
    if ((scale[1] == 0) && (scale[2] == 0))
    {
        *xppm = ABS(scale[0]);
        *yppm = ABS(scale[3]);
        *tan_s = (FS_FIXED)0;
        return SUCCESS;
    }

    s00 = scale[0];
    s01 = scale[1];
    s10 = scale[2];
    s11 = scale[3];

    /* too large ? ... > 16384 pixels per em */
    lim = (16384L << 16);
    if (ABS(s00) >= lim || ABS(s01) >= lim || ABS(s10) >= lim || ABS(s11) >= lim)
        return ERR_SCALE_LIMIT;

    /* check for degenerate transformation matrix */
    if ((s00 == 0 && s01 == 0) || (s10 == 0 && s11 == 0))
        return ERR_SCALE_DEGENERATE;
    if (FixDiv(s00, s10) == FixDiv(s01, s11))
        return ERR_SCALE_DEGENERATE;


    v.x = -s10;
    v.y = s00;
    fixed_norm(&v);

    *xppm = FixMul(s00, v.y) - FixMul(s10, v.x);
    *yppm = FixMul(s01, v.x) + FixMul(s11, v.y);
    *xppm = ABS(*xppm); /* only magnitude can be determined */
    *yppm = ABS(*yppm);
    *tan_s = FixMul( v.y , s01) - FixMul(v.x , s11);
    *tan_s = FixDiv(*tan_s, *yppm); /* sign is correct */

    return SUCCESS;
}

/****************************************************************/
/* modify_scale() adds additional skew to the input scale       */
FS_LONG modify_scale(FS_FIXED *scale, FS_FIXED tan_i_a)
{
    FS_FIXED xppm, yppm;
    FS_FIXED tan_s, tan_s_ia;
    FS_FIXED lambda, tan_x;
    FS_LONG err;

    /* decode the scale to get sizes and skew used to make scale */
    err = get_scale_inputs(scale, &xppm, &yppm, &tan_s);
    if (err != SUCCESS)
        return err;

    if (tan_s)
    {
        tan_s_ia = FixMul(tan_s, tan_i_a );
        tan_s_ia = FixDiv(tan_s + tan_i_a, FIXED_ONE - tan_s_ia) ;
    }
    else
        tan_s_ia = tan_i_a;

    tan_x = tan_s_ia - tan_s;
    lambda = FixMul(yppm, tan_x);
    lambda = FixDiv(lambda, xppm);

    scale[1] += FixMul(scale[0], lambda);
    scale[3] += FixMul(scale[2], lambda);
    return SUCCESS;
}

FS_LONG check_scale(_DS_ TTF_HEAD *head, FS_FIXED s00, FS_FIXED s01, FS_FIXED s10, FS_FIXED s11)
{
    FS_FIXED lim, limg, xppm, yppm, tan_s, lpm, x, y;
    FS_FIXED f00, f01, f10, f11, scale[4];
    FS_ULONG error;

    STATE.flags &=  ~ FLAGS_GRAY_LIMIT;

    scale[0] = s00;
    scale[1] = s01;
    scale[2] = s10;
    scale[3] = s11;

    error = get_scale_inputs(scale, &xppm, &yppm, &tan_s);
    if (error)
        return STATE.error = error;

    /* customer request - rasterize down to 1 PPEM (for greeking purposes) */
    lim = 1L << 16;

    if (xppm < lim || yppm < lim)
        return ERR_SCALE_LIMIT;

    /* too large ? ... > 16384 pixels per em */
    lim = (16384L << 16);
    limg = (4096L << 16);
    if (ABS(s00) >= lim || ABS(s01) >= lim || ABS(s10) >= lim || ABS(s11) >= lim)
        return ERR_SCALE_LIMIT;

    if (!head) return ERR_BAD_LFNT;

    /* convert to multipliers */
    lpm = (FS_FIXED)head->unitsPerEm << 16;
    f00 = FixDiv(s00, lpm);
    f01 = FixDiv(s01, lpm);
    f10 = FixDiv(s10, lpm);
    f11 = FixDiv(s11, lpm);

    /* scale the font bounding box ... recheck for size */
    x = f00 * head->xMin + f01 * head->yMin;
    y = f10 * head->xMin + f11 * head->yMin;
    if (ABS(x) >= lim || ABS(y) >= lim)
        return ERR_SCALE_LIMIT;
    if (ABS(x) >= limg || ABS(y) >= limg)
        STATE.flags |= FLAGS_GRAY_LIMIT;

    x = f00 * head->xMin + f01 * head->yMax;
    y = f10 * head->xMin + f11 * head->yMax;
    if (ABS(x) >= lim || ABS(y) >= lim)
        return ERR_SCALE_LIMIT;
    if (ABS(x) >= limg || ABS(y) >= limg)
        STATE.flags |= FLAGS_GRAY_LIMIT;

    x = f00 * head->xMax + f01 * head->yMin;
    y = f10 * head->xMax + f11 * head->yMin;
    if (ABS(x) >= lim || ABS(y) >= lim)
        return ERR_SCALE_LIMIT;
    if (ABS(x) >= limg || ABS(y) >= limg)
        STATE.flags |= FLAGS_GRAY_LIMIT;

    x = f00 * head->xMax + f01 * head->yMax;
    y = f10 * head->xMax + f11 * head->yMax;
    if (ABS(x) >= lim || ABS(y) >= lim)
        return ERR_SCALE_LIMIT;
    if (ABS(x) >= limg || ABS(y) >= limg)
        STATE.flags |= FLAGS_GRAY_LIMIT;

    /* so far so good */
    return SUCCESS;
}
/****************************************************************/
